/* implementacija hijerarhije klasa */
#include "SyntaxTree.hpp"

/* definisemo preoptereceni operator << kojim se stampa stablo */
std::ostream& operator <<(std::ostream& s, const SyntaxTreeNode& st) {

    st.Print(s);
    return s;
}

/************************************* Constant Node ***********************************************/

ConstantNode::ConstantNode(double val) {
    _value = val;
}
ConstantNode::ConstantNode(const ConstantNode& cn) {
    _value = cn._value;
}

void ConstantNode::Print(std::ostream& s) const {
    /* stampanje cvora je samo stampanje vrednosti */
    s << _value;
}
Funkcija* ConstantNode::Interpret(FuncSymbolTable& fs) const {
    /* rezultat interptetacije je konstantna funkcija sa datom vrednosti iz cvora */
    return new KonstantnaFunkcija(_value);
}
double ConstantNode::Calculate(FuncSymbolTable& fs, double x) const {
    /* vrednost cvora je vrednost konstante koja se u cvoru nalazi */
    return _value;
}

SyntaxTreeNode* ConstantNode::Clone() const {
    return new ConstantNode(*this);
}

/************************************* Constant Function Node ***********************************************/

ConstantFunctionNode::ConstantFunctionNode(double val) {
    _value = val;
}
ConstantFunctionNode::ConstantFunctionNode(const ConstantFunctionNode& cn) {
    _value = cn._value;
}

void ConstantFunctionNode::Print(std::ostream& s) const {
    /* stampanje cvora je samo stampanje vrednosti */
    s << _value;
}
Funkcija* ConstantFunctionNode::Interpret(FuncSymbolTable& fs) const {
    /* rezultat interptetacije je konstantna funkcija sa datom vrednosti iz cvora */
    return new KonstantnaFunkcija(_value);
}
double ConstantFunctionNode::Calculate(FuncSymbolTable& fs, double x) const {
    /* vrednost cvora je vrednost konstante koja se u cvoru nalazi */
    return _value;
}
SyntaxTreeNode* ConstantFunctionNode::Clone() const {

    return new ConstantFunctionNode(*this);
}

/************************************* Identifier Node ***********************************************/

IdentifierNode::IdentifierNode(const std::string& s) {
    _name = s;
}
IdentifierNode::IdentifierNode(const IdentifierNode& cn) {
    _name = cn._name;
}

void IdentifierNode::Print(std::ostream& s) const {
    /* stampanje cvora je stampanje naziva promenljive u cvoru */
    s << _name;
}
Funkcija* IdentifierNode::Interpret(FuncSymbolTable& fs) const {

    /* interpretacija cvora je pronalazenje funkcije u tablici simbola */

    /* pronalazimo simbol u tablici */
    Funkcija* f = fs.GetVariable(_name);

    /* ako simbol ne postoji, bacamo izuzetak */
    if (f == nullptr)
        throw std::invalid_argument("Variable not defined");

    /* ako smo nasli, vracamo kopiju funkcije iz tablice  */
    return f;
}
double IdentifierNode::Calculate(FuncSymbolTable& fs, double x) const {

    /* vrednost cvora je vrednost funkcije datog imena iz tablice u tacki x */

    /* pronalazimo simbol u tablici */
    Funkcija* f = fs.GetVariable(_name);

     /* ako simbol ne postoji, bacamo izuzetak */
    if (f == nullptr)
        throw std::invalid_argument("Variable not defined");

    /* ako smo nasli, racunamo vrednost funkcije u tacki */
    double rez = f->Value(x);
    /* unistavamo kopiju funkcije */
    delete f;
    /* vracamo rezultat */
    return rez;
}
SyntaxTreeNode* IdentifierNode::Clone() const {

    return new IdentifierNode(*this);
}

/************************************* Identity Node ***********************************************/

void IdentityNode::Print(std::ostream& s) const {
    s << "x";
}
Funkcija* IdentityNode::Interpret(FuncSymbolTable& fs) const {
    /* kao rezultat interpretacije vracamo identicku funkciju */
    return new IdentitetFunkcija();
}
double IdentityNode::Calculate(FuncSymbolTable& fs, double x) const {
    /* vrednost cvora je prosledjena vrednost */
    return x;
}
SyntaxTreeNode* IdentityNode::Clone() const {
    return new IdentityNode(*this);
}

/************************************* Function Node ***********************************************/

FunctionNode::FunctionNode(SyntaxTreeNode * expr) {
    _expr = expr;
}
FunctionNode::~FunctionNode() {
    delete _expr;
}

/************************************* Composition Node ***********************************************/

CompositionNode::CompositionNode(const std::string& s, SyntaxTreeNode* expr)
    : FunctionNode(expr) {
        _name = s;
    }
CompositionNode::CompositionNode(const CompositionNode& cn) 
    : FunctionNode(cn._expr->Clone()){
        _name = cn._name;
}

void CompositionNode::Print(std::ostream& s) const {
    s << _name << "(" << (*_expr) << ")";
}
Funkcija* CompositionNode::Interpret(FuncSymbolTable& fs) const {

    /* pronalazimo simbol u tablici */
    Funkcija* f = fs.GetVariable(_name);
    
    /* ako smo nasli, racunamo vrednost funkcije u tacki */
    if (f == nullptr)
        throw std::invalid_argument("Variable not defined");

    /* racunamo kompoziciju funkcije sa imenom _name sa funkcijom koja je rezultat interpretacije podstabla */
    Funkcija* rez = f->Compose(_expr->Interpret(fs));
    /* unistavamo kopiju promenljive */
    delete f;

    /* vracamo rezultat */
    return rez;
}
double CompositionNode::Calculate(FuncSymbolTable& fs, double x) const {

    /* pronalazimo simbol u tablici */
    Funkcija* f = fs.GetVariable(_name);
    
    /* ako smo nasli, racunamo vrednost funkcije u tacki */
    if (f == nullptr)
        throw std::invalid_argument("Variable not defined");

    /* odredjujemo vrednost kompozicije u tacki x */
    double rez = f->Value(_expr->Interpret(fs)->Value(x));

    /* unistavamo kopiju promenljive */
    delete f;

    /* vracamo rezultat */
    return rez;
}
SyntaxTreeNode* CompositionNode::Clone() const {

    return new CompositionNode(*this);
}

/************************************* Pow Node ***********************************************/

PowNode::PowNode(SyntaxTreeNode* expr, SyntaxTreeNode* step) 
    : FunctionNode(expr){
        _stepen = step;
}
PowNode::PowNode(const PowNode& cn) 
    : FunctionNode(cn._expr->Clone()){
    _stepen = cn._stepen->Clone();
}
PowNode::~PowNode() {
    delete _stepen;
}

void PowNode::Print(std::ostream& s) const {

    s << "pow("<< (*_expr) << "," << (*_stepen) << ")";
}
Funkcija* PowNode::Interpret(FuncSymbolTable& fs) const { 
    /* rezultat interpretacije je nova stepena funkcija
     * ovde samo treba obratiti paznju da je drugi argument konstruktora realan broj
     * pa zbog toga moramo da izracunamo vrednost _stepen podstabla nakon interpretacije
     * ovaj pristup je semanticki ispravan, jer prema konstrukciji gramatike _stepen ce uvek
     * biti konstantna funkcija. 
     */
    return new PowFunkcija(_expr->Interpret(fs), _stepen->Interpret(fs)->Value(0));
}
double PowNode::Calculate(FuncSymbolTable& fs, double x) const {
    /* kao rezultat vracamo vrednost stepene funckije */
    return pow(_expr->Interpret(fs)->Value(x), _stepen->Interpret(fs)->Value(x));
}
SyntaxTreeNode* PowNode::Clone() const {

    return new PowNode(*this);
}

/************************************* Sin Node ***********************************************/

SinNode::SinNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
SinNode::SinNode(const SinNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void SinNode::Print(std::ostream& s) const {
    s << "sin(" << (*_expr) << ")";
}
Funkcija* SinNode::Interpret(FuncSymbolTable& fs) const {
    return new SinFunkcija(_expr->Interpret(fs));
}
double SinNode::Calculate(FuncSymbolTable& fs, double x) const {
    return sin(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* SinNode::Clone() const {
    return new SinNode(*this);
}

/************************************* Cos Node ***********************************************/

CosNode::CosNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
CosNode::CosNode(const CosNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void CosNode::Print(std::ostream& s) const {
    s << "cos(" << (*_expr) << ")";
}
Funkcija* CosNode::Interpret(FuncSymbolTable& fs) const {
    return new CosFunkcija(_expr->Interpret(fs));
}
double CosNode::Calculate(FuncSymbolTable& fs, double x) const {
    return cos(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* CosNode::Clone() const {
    return new CosNode(*this);
}

/************************************* Tan Node ***********************************************/

TanNode::TanNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
TanNode::TanNode(const TanNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void TanNode::Print(std::ostream& s) const {
    s << "tan(" << (*_expr) << ")";
}
Funkcija* TanNode::Interpret(FuncSymbolTable& fs) const {
    return new TanFunkcija(_expr->Interpret(fs));
}
double TanNode::Calculate(FuncSymbolTable& fs, double x) const {
    return tan(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* TanNode::Clone() const {
    return new TanNode(*this);
}

/************************************* Cot Node ***********************************************/

CotNode::CotNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
CotNode::CotNode(const CotNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void CotNode::Print(std::ostream& s) const {
    s << "cot(" << (*_expr) << ")";
}
Funkcija* CotNode::Interpret(FuncSymbolTable& fs) const {
    return new CotFunkcija(_expr->Interpret(fs));
}
double CotNode::Calculate(FuncSymbolTable& fs, double x) const {
    return 1/tan(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* CotNode::Clone() const {
    return new CotNode(*this);
}

/************************************* Log Node ***********************************************/

LogNode::LogNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
LogNode::LogNode(const LogNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void LogNode::Print(std::ostream& s) const {
    s << "log(" << (*_expr) << ")";
}
Funkcija* LogNode::Interpret(FuncSymbolTable& fs) const {
    return new LogFunkcija(_expr->Interpret(fs));
}
double LogNode::Calculate(FuncSymbolTable& fs, double x) const {
    return log(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* LogNode::Clone() const {
    return new LogNode(*this);
}

/************************************* Exp Node ***********************************************/

ExpNode::ExpNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
ExpNode::ExpNode(const ExpNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void ExpNode::Print(std::ostream& s) const {
    s << "exp(" << (*_expr) << ")";
}
Funkcija* ExpNode::Interpret(FuncSymbolTable& fs) const {
    return new ExpFunkcija(_expr->Interpret(fs));
}
double ExpNode::Calculate(FuncSymbolTable& fs, double x) const {
    return log(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* ExpNode::Clone() const {
    return new ExpNode(*this);
}

/************************************* Sqrt Node ***********************************************/

SqrtNode::SqrtNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
SqrtNode::SqrtNode(const SqrtNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void SqrtNode::Print(std::ostream& s) const {
    s << "sqrt(" << (*_expr) << ")";
}
Funkcija* SqrtNode::Interpret(FuncSymbolTable& fs) const {
    return new SqrtFunkcija(_expr->Interpret(fs));
}
double SqrtNode::Calculate(FuncSymbolTable& fs, double x) const {
    return sqrt(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* SqrtNode::Clone() const {
    return new SqrtNode(*this);
}

/************************************* Suprotni Node ***********************************************/

SuprotniNode::SuprotniNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
SuprotniNode::SuprotniNode(const SuprotniNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void SuprotniNode::Print(std::ostream& s) const {
    s << "-(" << (*_expr) << ")";
}
Funkcija* SuprotniNode::Interpret(FuncSymbolTable& fs) const {
    return new SuprotnaFunkcija(_expr->Interpret(fs));
}
double SuprotniNode::Calculate(FuncSymbolTable& fs, double x) const {
    return -(_expr->Interpret(fs)->Value(x));
}
SyntaxTreeNode* SuprotniNode::Clone() const {
    return new SuprotniNode(*this);
}

/************************************* Diff Node ***********************************************/

DiffNode::DiffNode(SyntaxTreeNode* expr) 
    : FunctionNode(expr){

}
DiffNode::DiffNode(const DiffNode& cn) 
    : FunctionNode(cn._expr->Clone()){

}

void DiffNode::Print(std::ostream& s) const {
    s << "(" << (*_expr) << ")'";
}
Funkcija* DiffNode::Interpret(FuncSymbolTable& fs) const {
    /* rezultat interpretacije cvora je izvod podstabla od kojeg zavisi */
    return _expr->Interpret(fs)->Diff();
}
double DiffNode::Calculate(FuncSymbolTable& fs, double x) const {
    /* racunamo izvod */
    Funkcija* f = _expr->Interpret(fs)->Diff();
    /* odredjujemo vrednost izvoda u tacki */
    double rez = f->Value(x);
    /* unistavamo pomocni objekat */
    delete f;

    /* vracamo rezultat */
    return rez;
}
SyntaxTreeNode* DiffNode::Clone() const {
    return new DiffNode(*this);
}

/************************************* Value Node ***********************************************/

ValueNode::ValueNode(SyntaxTreeNode* expr, SyntaxTreeNode* x) 
    : FunctionNode(expr){
        _x = x;
}
ValueNode::ValueNode(const ValueNode& cn) 
    : FunctionNode(cn._expr->Clone()){
        _x = cn._x->Clone();
}
ValueNode::~ValueNode() {
    delete _x;
}

void ValueNode::Print(std::ostream& s) const {
    s << (*_expr) << "[" << (*_x) <<  "]";
}
Funkcija* ValueNode::Interpret(FuncSymbolTable& fs) const {
    /* rezultat interpretacije je konstantna funkcija */
    return new KonstantnaFunkcija(_expr->Interpret(fs)->Value(_x->Interpret(fs)->Value(0)));
}
double ValueNode::Calculate(FuncSymbolTable& fs, double x) const {

    return _expr->Interpret(fs)->Value(_x->Interpret(fs)->Value(x));
}
SyntaxTreeNode* ValueNode::Clone() const {
    return new ValueNode(*this);
}

/************************************* Binary Node ***********************************************/

BinaryNode::BinaryNode(const std::string _s, SyntaxTreeNode* l, SyntaxTreeNode* r) {
    _left = l;
    _right = r;
    _symbol = _s;
}
BinaryNode::BinaryNode(const BinaryNode& bo) {
    _left = bo._left->Clone();
    _right = bo._right->Clone();
    _symbol = bo._symbol;
}
BinaryNode::~BinaryNode() {
    delete _left;
    delete _right;
}

void BinaryNode::Print(std::ostream& s) const {

    s << (*_left) << _symbol << (*_right);
}
Funkcija* BinaryNode::Interpret(FuncSymbolTable& fs) const {
    /* u zavisnosti od simbola operacije izvrsavamo odgovarajucu operaciju  */
    if (_symbol == "+") {
        return new PlusFunkcija(_left->Interpret(fs), _right->Interpret(fs));
    }
    else if (_symbol == "-") {
        return new MinusFunkcija(_left->Interpret(fs), _right->Interpret(fs));
    }
    else if (_symbol == "*") {
        return new PutaFunkcija(_left->Interpret(fs), _right->Interpret(fs));
    }
    else if (_symbol == "/") {
        return new PodeljenoFunkcija(_left->Interpret(fs), _right->Interpret(fs));
    }
    return nullptr;
}
double BinaryNode::Calculate(FuncSymbolTable& fs, double x) const {

    if (_symbol == "+") {
        return _left->Interpret(fs)->Value(x) + _right->Interpret(fs)->Value(x);
    }
    else if (_symbol == "-") {
        return _left->Interpret(fs)->Value(x) - _right->Interpret(fs)->Value(x);
    }
    else if (_symbol == "*") {
        return _left->Interpret(fs)->Value(x) * _right->Interpret(fs)->Value(x);
    }
    else if (_symbol == "/") {
        return _left->Interpret(fs)->Value(x) / _right->Interpret(fs)->Value(x);
    }
    return 0;
}
SyntaxTreeNode* BinaryNode::Clone() const {
    return new BinaryNode(*this);
}

/************************************* Assignment Node ***********************************************/

AssignmentNode::AssignmentNode(const std::string& s, SyntaxTreeNode* expr) {
    _name = s;
    _expr = expr;
}
AssignmentNode::AssignmentNode(const AssignmentNode& an) {
    _name = an._name;
    _expr = an._expr->Clone();
}
AssignmentNode::~AssignmentNode() {
    delete _expr;
}

void AssignmentNode::Print(std::ostream& s) const {
    s << _name << " = " << (*_expr) << std::endl;
}
Funkcija* AssignmentNode::Interpret(FuncSymbolTable& fs) const {
    
    /* rezultat operacije je definicija promenljive */
    Funkcija* f = _expr->Interpret(fs);
    fs.DefineVariable(_name, f);
    
    /* vracamo novu funkciju kao bocni efekat */
    return f;
}
double AssignmentNode::Calculate(FuncSymbolTable& fs, double x) const {
    
    /* rezultat operacije je definicija promenljive */
    Funkcija* f = _expr->Interpret(fs);
    fs.DefineVariable(_name, f);
    
    /* vracamo vrednost funkcije kao bocni efekat */
    return f->Value(x);
}
SyntaxTreeNode* AssignmentNode::Clone() const {

    return new AssignmentNode(*this);
}

/************************************* Print Node ***********************************************/

PrintNode::PrintNode(SyntaxTreeNode* expr) {

    _expr = expr;
}
PrintNode::PrintNode(const PrintNode& an) {
    _expr = an._expr->Clone();
}
PrintNode::~PrintNode() {
    delete _expr;
}

void PrintNode::Print(std::ostream& s) const {

    s << "print(" << (*_expr) << ")";
}
Funkcija* PrintNode::Interpret(FuncSymbolTable& fs) const {
    /* rezultat interpretacije je stampanje funkcije na stdout */
    Funkcija* f = _expr->Interpret(fs);
    std::cout << (*f) << std::endl;
    return f;
}
double PrintNode::Calculate(FuncSymbolTable& fs, double x) const {

    Funkcija* f = _expr->Interpret(fs);
    std::cout << f->Value(x) << std::endl;
    return f->Value(x);
}
SyntaxTreeNode* PrintNode::Clone() const {

    return new PrintNode(*this);
}

/************************************* Sequence Node ***********************************************/

SequenceNode::SequenceNode(const SequenceNode& sn) {

    for (auto it = sn._statements.begin(); it != sn._statements.end(); it++) {
        _statements.push_back((*it)->Clone());
    }
}
SequenceNode::~SequenceNode() {
    for (auto it = _statements.begin(); it != _statements.end(); it++) {
        delete *it;
    }
}

void SequenceNode::Add(SyntaxTreeNode* stn) {
    _statements.push_back(stn);
}

void SequenceNode::Print(std::ostream& s) const {
    
    for (auto it = _statements.begin(); it != _statements.end(); it++) {
        (*it)->Print(s);
        s << std::endl;
    }
}
Funkcija* SequenceNode::Interpret(FuncSymbolTable& fs) const {

    /* rezultat interpretacije je interpretacija svih cvorova koji su u vektoru */
    for (auto it = _statements.begin(); it != _statements.end(); it++) {
        (*it)->Interpret(fs);
    }
    return nullptr;
}
double SequenceNode::Calculate(FuncSymbolTable& fs, double x) const {

    for (auto it = _statements.begin(); it != _statements.end(); it++) {
        (*it)->Interpret(fs)->Value(x);
    }
    return 0;
}
SyntaxTreeNode* SequenceNode::Clone() const {

    return new SequenceNode(*this);
}

/************************************* Empty Node ***********************************************/

void EmptyNode::Print(std::ostream& s) const {
    s << std::endl;
}
Funkcija* EmptyNode::Interpret(FuncSymbolTable& fs) const {
    return nullptr;
}
double EmptyNode::Calculate(FuncSymbolTable& fs, double x) const {
    return 0;
}
SyntaxTreeNode* EmptyNode::Clone() const {
    return new EmptyNode(*this);
}